<?php

namespace App\Controllers\Admin;

use App\Controllers\Admin\AdminMain;
use App\Models\ParticipantsModel;
use App\Models\ProductModel;
use App\Models\QuotModel;
use App\Models\UserModel;
use App\Models\ValuesModel;

class Quot extends AdminMain
{
    private $quot, $products, $participants, $values;

    public function __construct()
    {

        parent::__construct();

        $this->quot = new QuotModel();

        $this->participants = new ParticipantsModel();

        $this->products = new ProductModel();

        $this->values = new ValuesModel();
    }

    // Cria a cotação e adiciona os produtos.

    public function create()
    {
        $id_store = $this->request->getPost('id_store');

        $validationRule = [
            'quot' => [
                'label' => 'Text File',
                'rules' => [
                    'uploaded[quot]',
                    'ext_in[quot,csv]',
                ],
            ],
        ];

        if (! $this->validateData([], $validationRule)) {

            return redirect()->to('a/store/details/' . $id_store)->with('error', 'Escolha um arquivo.csv válido.');
        }

        $id_quot = $this->quot->createQuot($id_store);

        if (!$id_quot) {

            return redirect()->to('a/store/details/' . $id_store)->with('error', 'Erro ao adicionar a cotação.');
        }

        if (!$this->products->createProduct($id_quot)) {

            $this->products->where('cotacao_idcotacao', $id_quot)->delete();

            $this->quot->delete($id_quot);

            return redirect()->to('a/store/details/' . $id_store)->with('error', 'Erro ao adicionar produtos na cotação.');
        }

        return redirect()->to('a/store/details/' . $id_store);
    }

    // Modifica o status da cotação: Aberta, fechada, enviada.

    public function edit($status, $id_quot, $id_store)
    {
        $link = ($status == 'e') ? 'a/order/orders_by_quot/' : 'a/quot/details/';

        if (!in_array($status, ['a', 'e', 'f'])) {

            return redirect()->to($link . $id_quot . '/' . $id_store . $this->page)->with('error', 'Status incorreto.');
        }

        if (!$this->quot->update($id_quot, ['status' => $status])) {

            return redirect()->to($link . $id_quot . '/' . $id_store . $this->page)->with('error', 'Erro ao mudar o status da cotação.');
        }

        return redirect()->to($link . $id_quot . '/' . $id_store . $this->page);
    }

    // Mostra todos os detalhes dos produtos, participantes, valores e vencedores da cotação.

    public function details($id_quot, $id_store)
    {
        /**
         * $result_per_page: Define a quantidade de produtos por página.
         */

        $result_per_page = 60;

        $suppliers = new UserModel();

        $status_quot = $this->quot->where('idcotacao', $id_quot)->findColumn('status');

        $get_participants = $this->participants->where(['cotacao' => $id_quot])->findAll();

        $get_products = $this->products->where(['cotacao_idcotacao' => (int)$id_quot])->paginate($result_per_page);

        $first_product = $get_products[0]->idproduto;

        $count_winners = $this->products->where(['cotacao_idcotacao' => $id_quot, 'vencedor >' => 0])->selectCount('vencedor')->find();

        return view('index', [

            'title_complement' => 'Cotação da loja ' . $this->nameStore($id_store),

            $this->CONTENT_NORMAL => view('admin/quot/details_quot', [

                'id_store' => $id_store,

                'id_quot' => $id_quot,

                'status_quot' => $status_quot[0],

                'values' => $this->values->where([
                    'idcotacao' => $id_quot,
                    'idproduto >=' => (int)$first_product,
                    'idproduto <=' => ($first_product + $result_per_page),
                    'valor >' => 0,
                ])->findAll(),

                'suppliers' => $suppliers->where('tipo', 'f')->findAll(),

                'participants' => $get_participants,

                'products' => $get_products,

                'pager' => $this->products->pager,

                'count_winners' => $count_winners[0]->vencedor,

            ]),

        ]);
    }

    // Adiciona um fornecedor à cotação.

    public function addSupplier($id_quot, $id_store, $id_supplier)
    {

        if (!$this->participants->save(['cotacao' => $id_quot, 'usuario' => $id_supplier, 'concluido' => 'n'])) {

            return redirect()->to('a/quot/details/' . $id_quot . '/' . $id_store . $this->page)->with('error', 'Erro ao adicionar o participante.');
        }

        return redirect()->to('a/quot/details/' . $id_quot . '/' . $id_store . $this->page);
    }

    // Remove o participante da cotação.

    public function removeSupplier($id_quot, $id_store, $id_supplier)
    {

        if (!$this->participants->where(['cotacao' => $id_quot, 'usuario' => $id_supplier])->delete()) {

            return redirect()->to('a/quot/details/' . $id_quot . '/' . $id_store . $this->page)->with('error', 'Erro ao remover o participante.');
        }

        if (!$this->values->where(['idcotacao' => $id_quot, 'idusuario' => $id_supplier])->delete()) {

            return redirect()->to('a/quot/details/' . $id_quot . '/' . $id_store . $this->page)->with('error', 'Erro ao remover os valores do participante.');
        }

        return redirect()->to('a/quot/details/' . $id_quot . '/' . $id_store . $this->page);
    }

    // O administrador seleciona o valor.

    public function selectValue($id_product, $id_participant, $id_quot, $id_store)
    {

        if (!$this->products->update(['idproduto' => $id_product, 'cotacao_idcotacao' => $id_quot], ['vencedor' => $id_participant])) {

            return redirect()->to('a/quot/details/' . $id_quot . '/' . $id_store . $this->page)->with('error', 'Erro ao selecionar o valor.');
        }

        return redirect()->to('a/quot/details/' . $id_quot . '/' . $id_store . $this->page . '#' . $id_product . $id_participant);
    }

    // O administrador remove o valor selecionado.

    public function removeValue($id_product, $id_participant, $id_quot, $id_store)
    {

        if (!$this->products->update(['idproduto' => $id_product, 'cotacao_idcotacao' => $id_quot], ['vencedor' => ''])) {

            return redirect()->to('a/quot/details/' . $id_quot . '/' . $id_store . $this->page)->with('error', 'Erro ao remover o valor.');
        }

        return redirect()->to('a/quot/details/' . $id_quot . '/' . $id_store . $this->page . '#' . $id_product . $id_participant);
    }

    //   Seleciona automaticamente o menor valor de cada produto.

    public function selectLowestPrice($id_quot, $id_store)
    {

        if (!$this->products->selectLowestPrice($id_quot)) {

            return redirect()->to('a/quot/details/' . $id_quot . '/' . $id_store . $this->page)->with('error', 'Erro ao selecionar o menor valor.');
        }

        return redirect()->to('a/quot/details/' . $id_quot . '/' . $id_store . $this->page);
    }

    // Edita o campo quantidade.

    public function editQtd()
    {

        $qtd = $this->request->getPost('qtd');
        $id_quot = $this->request->getPost('id_quot');
        $id_store = $this->request->getPost('id_store');
        $id_product = $this->request->getPost('id_product');

        if (!$this->products->update(['idproduto' => $id_product, 'cotacao_idcotacao' => $id_quot], ['quantidade' => $qtd])) {

            return redirect()->to('a/quot/details/' . $id_quot . '/' . $id_store . $this->page)->with('error', 'Erro ao atualizar a quantidade.');
        }

        return redirect()->to('a/quot/details/' . $id_quot . '/' . $id_store . $this->page);
    }

    /**
     * 
     * Função para mudar o status do produto.
     * 
     * O administrador decidirá se o produto será cotado.
     * 
     * Para evitar mudanças no banco de dados foi decidido que o status será atualizado no campo "quantidade", da tabela produto.
     * 
     * Funciona da seguinte forma: Quando o administrador desativar um produto, será adicionado no campo "quantidade" o sinal de menos (-)
     * 
     * junto com o número referente à quantidade, ficando assim um número negativo.
     * 
     * Essa informação será tratada para evitar que o produto não apareça na cotação, do lado dos fornecedores participantes.
     * 
     * Dessa forma se preservará o número referente à quantidade.
     * 
     * Se o administrador optar por adicionar (ativar) o produto na cotação, a função removerá o sinal de menos, mantendo o número original do campo "quantidade". 
     * 
     */

    public function changeStatusProduct($new_status, $id_product, $id_quot, $id_store)
    {

        if (!$this->products->update(['idproduto' => $id_product, 'cotacao_idcotacao' => $id_quot], ['quantidade' => $new_status])) {

            return redirect()->to('a/quot/details/' . $id_quot . '/' . $id_store . $this->page)->with('error', 'Erro ao mudar o status do produto.');
        }

        return redirect()->to('a/quot/details/' . $id_quot . '/' . $id_store . $this->page);
    }

    /**
     * Função para mostrar a evolução dos participantes.
     * 
     * Retorna um array com o percentual de cada participante, conforme insere valores nos produtos.
     * 
     * A referência é no avanço das linhas da tabela, não na quantidade de valores adicionados.
     * 
     * O percetual avança conforme se aproxima da útima linha da tabela.
     */

    public function progressByParticipant()
    {

        $id_quot = $this->request->getPost('id_quot');

        $participants = $this->participants->where('cotacao', $id_quot)->findAll();

        $min_product = $this->products
            ->selectMin('idproduto')
            ->where('cotacao_idcotacao', $id_quot)
            ->find();

        $qtd_product = $this->products
            ->selectCount('idproduto')
            ->where('cotacao_idcotacao', $id_quot)
            ->find();

        foreach ($participants as $participant) {

            $max_value_user = $this->values
                ->selectMax('idproduto')
                ->where(['idcotacao' => $id_quot, 'idusuario' => $participant->usuario, 'valor >' => 1])
                ->find();

            $position_user = ($max_value_user[0]->idproduto - ($min_product[0]->idproduto - 1));

            $percent_user = ($position_user / $qtd_product[0]->idproduto) * 100;

            $percent_user = ($percent_user > 0) ? $percent_user : 0;

            /**
             * Linha para mostrar a barra com 100%, caso o participante tenha concluído.
             */

            // $percent_user = ($participant->concluido == 's') ? 100 : $percent_user;

            $progres_user[] = array('id_participant' => $participant->usuario, 'percent_user' => round($percent_user, 1));
        }

        return json_encode(['progress' => $progres_user]);
    }

    public function delete($id_quot, $id_store)
    {

        if (!$this->participants->where(['cotacao' => $id_quot])->delete()) {

            return redirect()->to('a/quot/details/' . $id_quot . '/' . $id_store . $this->page)->with('error', 'Erro ao excluir os participantes da cotação.');
        }

        if (!$this->values->where(['idcotacao' => $id_quot])->delete()) {

            return redirect()->to('a/quot/details/' . $id_quot . '/' . $id_store . $this->page)->with('error', 'Erro ao remover os valores dos participantes da cotação.');
        }

        if (!$this->products->where(['cotacao_idcotacao' => $id_quot])->delete()) {

            return redirect()->to('a/quot/details/' . $id_quot . '/' . $id_store . $this->page)->with('error', 'Erro ao excluir os produtos da cotação.');
        }

        if (!$this->quot->where(['idcotacao' => $id_quot])->delete()) {

            return redirect()->to('a/quot/details/' . $id_quot . '/' . $id_store . $this->page)->with('error', 'Erro ao excluir a cotação.');
        }

        return redirect()->to('a/store/details/' . $id_store . $this->page);
    }
}
